#include <iostream>
#include <time.h>
#include <algorithm>
#include <iterator>
#include <iomanip>
#include <sstream>
#include "Yolo_lib.h"

using namespace cv;
using namespace std;
using namespace dnn;

// Remove the bounding boxes with low confidence using non-maxima suppression
void post_process(Mat& frame, const vector<Mat>& outs, float conf_threshold, float nms_threshold, vector<std::string> classes, vector<cv::Scalar> colors)
{
	vector<int> classIds;
	vector<float> confidences;
	vector<Rect> boxes;
	int frame_width = frame.cols;
	int frame_height = frame.rows;

	CV_Assert(outs.size() > 0);

	for (size_t i = 0; i < outs.size(); ++i)
	{
		// Scan through all the bounding boxes output from the network and keep only the
		// ones with high confidence scores. Assign the box's class label as the class
		// with the highest score for the box.
		float* data = (float*)outs[i].data;
		for (int j = 0; j < outs[i].rows; ++j, data += outs[i].cols)
		{
			Mat scores = outs[i].row(j).colRange(5, outs[i].cols);
			Point classIdPoint;
			double confidence;
			// Get the value and location of the maximum score
			minMaxLoc(scores, 0, &confidence, 0, &classIdPoint);

			if (confidence > conf_threshold)
			{
				int centerX = (int)(data[0] * frame_width);
				int centerY = (int)(data[1] * frame_height);
				int width = (int)(data[2] * frame_width);
				int height = (int)(data[3] * frame_height);
				int x = centerX - (width / 2);
				int y = centerY - (height / 2);

				boxes.push_back(Rect(x, y, width, height));
				confidences.push_back((float)confidence);
				classIds.push_back(classIdPoint.x);
			}
		}
	}

	// Perform non maximum suppression to eliminate redundant overlapping boxes with
	// lower confidences
	vector<int> indices;
	cv::dnn::NMSBoxes(boxes, confidences, conf_threshold, nms_threshold, indices);
	if (indices.size() > 0) {
		for (size_t i = 0; i < indices.size(); ++i)
		{
			int idx = indices[i];
			Rect box = boxes[idx];
			draw_box(frame, classIds[idx], confidences[idx], box, classes, colors);
		}
	}
}

// Get the names of the output layers
vector<String> getOutputsNames(const Net& net)
{
	static vector<String> names;
	if (names.empty())
	{
		//Get the indices of the output layers, i.e. the layers with unconnected outputs
		vector<int> outLayers = net.getUnconnectedOutLayers();

		//get the names of all the layers in the network
		vector<String> layersNames = net.getLayerNames();

		// Get the names of the output layers in names
		names.resize(outLayers.size());
		for (size_t i = 0; i < outLayers.size(); ++i)
			names[i] = layersNames[outLayers[i] - 1];
	}
	return names;
}

// this function is used to get the extension of the file
string getFileExt(const string& s) {

	size_t i = s.rfind('.', s.length());
	if (i != string::npos) {
		return(s.substr(i + 1, s.length() - i));
	}

	return("");
}

// print the vector of classes
void print_vector(std::vector<std::string> classes)
{
	// Print all the available classes
	std::cout << "Total number of classes: " << classes.size() << endl;
	std::cout << "Classes: " << endl;
	for (std::vector<string>::const_iterator i = classes.begin(); i != classes.end(); ++i)
		std::cout << *i << endl;
}

// Draw the predicted bounding box
void draw_box(Mat& frame, int classId, float conf, Rect box, vector<std::string> classes, vector<cv::Scalar> colors)
{
	if (!colors.empty()) {
		//Draw a rectangle displaying the bounding box
		cv::rectangle(frame, Point(box.x, box.y), Point(box.x + box.width, box.y + box.height), colors[classId], 2);
		CV_Assert(classes.size() == colors.size());
	}

	std::stringstream stream;
	stream << std::fixed << std::setprecision(2) << conf*100;
	std::string s = stream.str();

	//Get the label for the class name and its confidence
	string label;
	if (!classes.empty())
	{
		CV_Assert(classId < (int)classes.size());
		label = classes[classId] + ":" + s + "%";
	}

	//Display the label at the top of the bounding box
	int baseLine, top;
	Size labelSize = getTextSize(label, FONT_HERSHEY_SIMPLEX, 0.5, 1, &baseLine);
	top = max(box.y, labelSize.height);
	putText(frame, label, Point(box.x, top - 5), FONT_HERSHEY_SIMPLEX, 0.5, colors[classId], 2);
}